  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  '
  ' DDS Function Generator V1
  ' Geoff Graham, Jan 16
  '
  ' Requires MMBasic 5.2 or later and an ILI9341 based LCD panel with touch
  ' Plus an AD9833 DDS Function Generator module
  '
  ' Modified by Nicholas Vinen, July 2017 to add IF alignment sweep feature
  '
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  Option Explicit
  Option Default Integer
  Option AutoRun On
  
  If MM.HRes <> 320 Or MM.VRes <> 240 Then Error "Requires a 320 x 240 display"
  
  ' colours used in this program
  Const cTitle = RGB(yellow)          ' title colour
  Const cEntry = RGB(white)           ' colour used for the entry
  Const cButton = RGB(cyan)           ' the key colour
  Const cSave = RGB(white)            ' the save button
  Const cDel = RGB(magenta)           ' the delete button
  Const cSpecial = RGB(red)           ' special buttons
  Const cSelected = RGB(blue)         ' used to indicated selected button
  Const cBGround = RGB(black)         ' the background colour
  
  ' pin allocations
  Const CS = 17          ' pin for MCP41010 chip select (CS) pin on module
  Const FSY = 18         ' pin for AD9833 chip select(FSY) pin on module
  Const TrigPin = 24     ' pin for the sweep trigger output/RF detector measurement input
  CONST BiasPin = 22
  
  ' these variables are saved with VAR SAVE
  Dim Freq = 0, Level = 100, Mode = 0, ActiveDigit = 1, SineAM = 0
  Dim SweepStart = 1, SweepEnd = 9999999, SweepTime = 1, LogSweep = 0
  DIM CursorUpdateInterval = 1, FreqSpan = 20000, InvertVoltageReading = 1
  
  ' general global variables
  Dim i
  Dim TouchInt
  Dim DisplayLevel.Last
  
  ' IF alignment variables
  DIM FreqOffset = 0, FreqIter, FreqCount, FreqStarting, CursorUpdateCounter
  DIM FLOAT FreqIterScale, FreqIterMin, NewFreqIterScale, NewFreqIterMin, OldIFAlignData
  DIM FLOAT IFAlignData(80)
  DIM CursorX(2)
  DIM CursorNum
  CONST FreqNumAvgs = 3
  CONST AlignGraphTop = 24
  CONST AlignGraphHeight = 152
  
  ' arrays used to keep track of touch sensitive buttons
  Dim x1(30), y1(30), x2(30), y2(30)
  Dim Integer key_coord(17, 5)
  Dim String key_caption(17)
  
  ' position of the AM check box
  Const AM_Box_x = 8, AM_Box_y = 128
  CONST IF_Box_X = 8, IF_Box_y = 84
  
  ' start of the program
  CPU 48
  On Error Skip
  VAR Restore

  CLS cBGround
  
  ' setup the output pins
  Pin(CS) = 1
  SetPin CS, dout
  Pin(FSY) = 1
  SetPin FSY, dout
  Pin(TrigPin) = 0
  IF Mode = 4 THEN
    SetPin TrigPin, AIN
    FreqIter = 0
    FreqCount = 0
    FreqIterScale = 1.0
    FreqIterMin = 0.0
    NewFreqIterScale = 1.0
    NewFreqIterMin = 0.0
    UpdateIFAlign
    CursorUpdateCounter = CursorUpdateInterval-1
    IF InvertVoltageReading = 1 THEN
      SETPIN BiasPin, DOUT
      PIN(BiasPin) = 1
    ELSEIF InvertVoltageReading = 2 THEN
      SETPIN BiasPin, DOUT
      PIN(BiasPin) = 0
    ELSE
'      PIN(BiasPin) = 0
      SETPIN BiasPin, DIN
    ENDIF
  ELSE
    SetPin TrigPin, DOUT
  ENDIF
  
  ' setup the initial display
  DisplayMain
  
  ' main program loop
  ' this maintains the main screen
  Do
    ' if the mode is sinewave and AM selected then modulate the output
    If Mode = 0 And SineAM = 1 Then
      DoAM                                      ' this will only return when the screen is touched
    EndIf
    IF Mode = 4 THEN UpdateIFAlign
    i = CheckTouch(1, 23)                       ' scan all the buttons looking for a touch
    WatchDog 5000
    If i > 0 Then SetTick 60000, SaveState, 2   ' trigger a state save in one minute if a button has been touched
    Timer = 0                                   ' used for auto repeat
    Select Case i
      Case 1 To 7   ' a digit of the frequency has been selected
        ActiveDigit = 8 - i
        DisplayFreq 0, 0, ActiveDigit
        Do While Touch(x) <> -1
          If Timer > 1000 Then
            KeyPadInput "Freq (Hz)", "", "Zero Not Valid", 9999999, 1, Freq
            CLS cBGround
            DisplayMain
          EndIf
        Loop
      Case 8             ' the frequency up button has been selected
RepeatUp:
        WatchDog 5000
        i = Freq
        Freq = Freq + 10^(ActiveDigit - 1)
        If Timer > 1400 Then Freq = Freq + 10^(ActiveDigit - 1)
        If Timer > 3000 Then Freq = Freq + (10^(ActiveDigit - 1) * 4)
        If Freq > 9999999 Then Freq = i : Timer = 401
        DisplayFreq 1, 0, ActiveDigit
        SetFreq
        Do While Touch(x) <> -1
          If Timer > 400 Then GoTo RepeatUp
        Loop
        DisplayFreq 0, 0, ActiveDigit
        Pause 150
      Case 9             ' the frequency down button has been selected
RepeatDown:
        WatchDog 5000
        i = Freq
        Freq = Freq - 10^(ActiveDigit - 1)
        If Timer > 1400 Then Freq = Freq - 10^(ActiveDigit - 1)
        If Timer > 3000 Then Freq = Freq - (10^(ActiveDigit - 1) * 4)
        If Freq < 0 Then Freq = i : Timer = 401
        DisplayFreq 0, 1, ActiveDigit
        SetFreq
        Do While Touch(x) <> -1
          If Timer > 400 Then GoTo RepeatDown
        Loop
        DisplayFreq 0, 0, ActiveDigit
        Pause 150
      Case 10 To 13
        IF Mode = 4 THEN
          ClearTouchZones 0, 23
          SetPin TrigPin, DOUT
 '         PIN(BiasPin) = 0
          SETPIN BiasPin, DIN
          CLS
        ENDIF
        Mode = i - 10
        DisplayMain
        Do While Touch(x) <> -1 : WatchDog 5000 : Loop
      Case 14             ' the level down button has been selected
RepeatLevelDwn:
        WatchDog 5000
        Level = Level - 1
        If Level < 0 Then Level = 0
        DisplayLevel MM.HRes/2 + ((Mode = 0) * 60), 0, 1
        SetLevel
        Do While Touch(x) <> -1
          If Timer > 400 Then GoTo RepeatLevelDwn
        Loop
        DisplayLevel MM.HRes/2 + ((Mode = 0) * 60), 0, 0
        Pause 150
      Case 15             ' the level up button has been selected
RepeatLevelUp:
        WatchDog 5000
        Level = Level + 1
        If Level > 100 Then Level = 100
        DisplayLevel MM.HRes/2 + ((Mode = 0) * 60), 1, 0
        SetLevel
        Do While Touch(x) <> -1
          If Timer > 400 Then GoTo RepeatLevelUp
        Loop
        DisplayLevel MM.HRes/2 + ((Mode = 0) * 60), 0, 0
        Pause 150
      Case 16             ' the level number has been selected
        If Mode <> 2 Then
          KeyPadInput "Level (%)", "Max Is 100", "", 100, 0, Level
          CLS cBGround
          DisplayMain
        EndIf
      Case 17              ' the AM checkbox has been selected
        SineAM = Not SineAM
        Box AM_Box_x, AM_Box_y, 23, 23, 1, cEntry, cBGround
        If SineAM Then
          Line AM_Box_x + 3, AM_Box_y + 3, AM_Box_x + 19, AM_Box_y + 19, 1, cEntry
          Line AM_Box_x + 3, AM_Box_y + 19, AM_Box_x + 19, AM_Box_y + 3, 1, cEntry
        EndIf
        Do While Touch(x) <> -1 : WatchDog 5000 : Loop
      CASE 18               ' the IF align button has been pressed
        ClearTouchZones 0, 18
        x1(19) =   0 : x2(19) = 319 : y1(19) = 0 : y2(19) = AlignGraphTop + 20
        x1(20) =   0 : x2(20) = 100 : y1(20) = AlignGraphTop+AlignGraphHeight - 10 : y2(20) = AlignGraphTop+AlignGraphHeight+16+10
        x1(21) = 100 : x2(21) = 219 : y1(21) = AlignGraphTop+AlignGraphHeight - 10 : y2(21) = AlignGraphTop+AlignGraphHeight+16+10
        x1(22) = 219 : x2(22) = 319 : y1(22) = AlignGraphTop+AlignGraphHeight - 10 : y2(22) = AlignGraphTop+AlignGraphHeight+16+10
        x1(23) =   0 : x2(23) = 319 : y1(23) = AlignGraphTop + 20 : y2(23) = AlignGraphTop+AlignGraphHeight - 10
        Mode = 4
        PIN(TrigPin) = 0
        SetPin TrigPin, AIN
        FreqIter = 0
        FreqCount = 0
        FreqIterScale = 1.0
        FreqIterMin = 0.0
        NewFreqIterScale = 1.0
        NewFreqIterMin = 0.0
        UpdateIFAlign
        CursorUpdateCounter = CursorUpdateInterval-1
        IF InvertVoltageReading = 1 THEN
          SETPIN BiasPin, DOUT
          PIN(BiasPin) = 1
        ELSEIF InvertVoltageReading = 2 THEN
          SETPIN BiasPin, DOUT
          PIN(BiasPin) = 0
        ELSE
'          PIN(BiasPin) = 0
          SETPIN BiasPin, DIN
        ENDIF
        CLS
        DisplayMain
      CASE 19
        SELECT CASE CursorUpdateInterval
          CASE 1
            CursorUpdateInterval = 2
            VAR SAVE CursorUpdateInterval
          CASE 2
            CursorUpdateInterval = 4
            VAR SAVE CursorUpdateInterval
          CASE 4
            CursorUpdateInterval = 1
            VAR SAVE CursorUpdateInterval
        END SELECT
        Do While Touch(x) <> -1
        Loop
      CASE 20
        KeyPadInput "Span (Hz)", "", "Zero Not Valid", 100000, 1, FreqSpan
        VAR SAVE FreqSpan
        CLS cBGround
        DisplayMain
      CASE 21
        KeyPadInput "Freq (Hz)", "", "Zero Not Valid", 9999999, 1, Freq
        VAR SAVE Freq
        CLS cBGround
        DisplayMain
      CASE 22
        KeyPadInput "Span (Hz)", "", "Zero Not Valid", 100000, 1, FreqSpan
        VAR SAVE FreqSpan
        CLS cBGround
        DisplayMain
      CASE 23
        InvertVoltageReading = InvertVoltageReading + 1
        IF InvertVoltageReading > 2 THEN InvertVoltageReading = 0
        VAR SAVE InvertVoltageReading
        IF InvertVoltageReading = 1 THEN
          SETPIN BiasPin, DOUT
          PIN(BiasPin) = 1
        ELSEIF InvertVoltageReading = 2 THEN
          SETPIN BiasPin, DOUT
          PIN(BiasPin) = 0
        ELSE
'          PIN(BiasPin) = 0
          SETPIN BiasPin, DIN
        ENDIF
    End Select
  Loop
End
  
  
  ' display the main screen (sine, square, etc)
Sub DisplayMain
  IF Mode <> 4 THEN DisplayFreq 0, 0, ActiveDigit
  SetLevel
  SetFreq
  IF Mode = 4 THEN DrawModeButtons 0 ELSE DrawModeButtons Mode
  x1(17) = 0 : y1(17) = 0 : x2(17) = 0 : y2(17) = 0
  x1(18) = 0 : y1(18) = 0 : x2(18) = 0 : y2(18) = 0
  Box 0, 89, MM.HRes - 58, 29, 0, 0, cBGround
  Box 0, 118, MM.HRes, 54, 0, 0, cBGround
  
  IF Mode <> 0 THEN
    BOX IF_Box_x, IF_Box_y, 140, 5, 1, RGB(BLACK), RGB(BLACK) ' erase part of IF align box which would remain
    BOX AM_Box_x -2, AM_Box_y + 43, 90, 6, 1, RGB(BLACK), RGB(BLACK)
  ENDIF
  Select Case Mode
    Case 0
      Box AM_Box_x, AM_Box_y, 23, 23, 1, cEntry, cBGround
      If SineAM Then
        Line AM_Box_x + 3, AM_Box_y + 3, AM_Box_x + 19, AM_Box_y + 19, 1, cEntry
        Line AM_Box_x + 3, AM_Box_y + 19, AM_Box_x + 19, AM_Box_y + 3, 1, cEntry
      EndIf
      Text AM_Box_x + 33, AM_Box_y - 1, "AM", , 2, 1, cEntry, cBGround
      Text AM_Box_x - 9, AM_Box_y + 27, "(1", , 1, 2, cEntry, cBGround
      Text AM_Box_x + 29, AM_Box_y + 27, "kHz)", , 1, 2, cEntry, cBGround
      
      BOX IF_Box_x, IF_Box_y, 140, 30, 1, RGB(WHITE), RGB(192,0,192)
      Text IF_Box_x + 70, IF_Box_y + 15, "IF Align", CM, 1, 2, RGB(WHITE), RGB(192,0,192)
      
      x1(17) = 0 : y1(17) = AM_Box_y - 16 : x2(17) = AM_Box_x + 81 : y2(17) = AM_Box_y + 62
      x1(18) = IF_Box_x : y1(18) = IF_Box_y : x2(18) = IF_Box_x + 140 : y2(18) = IF_Box_y + 30
      DisplayLevel 220, 0, 0, 0
    Case 1
      DisplayLevel MM.HRes/2, 0, 0, 0
    Case 2
      DisplayLevel MM.HRes/2, 0, 0, 1
    Case 3
      Box 0, 0, MM.HRes, MM.VRes - 51, 0, 0, cBGround
      DisplaySweep
      Box 0, 0, MM.HRes, MM.VRes - 51, 0, 0, cBGround
      DisplayMain
    CASE 4
      IF Freq >= 1000000 THEN
        Text 0, 176, STR$((Freq - FreqSpan / 2) / 1000000.0, 1, 3) + "MHz", LT, 1, 1, cEntry, cBGround
        Text 160, 176, STR$(Freq / 1000000.0, 1, 3) + "MHz", CT, 1, 1, cEntry, cBGround
        Text 319, 176, STR$((Freq + FreqSpan / 2) / 1000000.0, 1, 3) + "MHz", RT, 1, 1, cEntry, cBGround
      ELSE
        Text 0, 176, STR$((Freq - FreqSpan / 2) / 1000.0, 1, 1) + "kHz", LT, 1, 1, cEntry, cBGround
        Text 160, 176, STR$(Freq / 1000.0, 1, 1) + "kHz", CT, 1, 1, cEntry, cBGround
        Text 319, 176, STR$((Freq + FreqSpan / 2) / 1000.0, 1, 1) + "kHz", RT, 1, 1, cEntry, cBGround
      ENDIF
      line 160, AlignGraphTop, 160, AlignGraphTop+AlignGraphHeight, 1, RGB(80,80,80)
  End Select
End Sub
  
  
  ' display the sub screen for the sweep functionality
  ' this will only return when another mode (sine, square, etc) is selected
Sub DisplaySweep
  Local i
  Const Log_Box_x = MM.HRes - 37, Log_Box_y = 130
  Local SweepTime.steps(5) = (50, 100, 200, 500, 1000, 2000)
  Do
    Text MM.HRes, 0, "Start", R, 1, 2, cTitle, cBGround
    Text MM.HRes - 8, 26, "(Hz)", R, 1, 2, cTitle, cBGround
    Text 0, 0, Str$(SweepStart, 7, 0), , 3, 1, cEntry, cBGround
    Text MM.HRes, 55, "End ", R, 1, 2, cTitle, cBGround
    Text MM.HRes - 8, 81, "(Hz)", R, 1, 2, cTitle, cBGround
    Text 0, 55, Str$(SweepEnd, 7, 0), , 3, 1, cEntry, cBGround
    Text 130, 107, "Sweep", L, 1, 2, cTitle, cBGround
    If SweepTime > 3 Then
      Text MM.HRes - 112, 134, Str$(SweepTime.steps(SweepTime)/1000, 3, 0), R, 3, 1, cEntry, cBGround
      Text MM.HRes - 64, 190, "S ", RB, 2, 1, cEntry, cBGround
    Else
      Text MM.HRes - 112, 134, Str$(SweepTime.steps(SweepTime), 3, 0), R, 3, 1, cEntry, cBGround
      Text MM.HRes - 64, 190, "YS", RB, 2, 1, cEntry, cBGround   ' Note: YS prints as mS
    EndIf
    Text 44, 107, "Level", C, 1, 2, cTitle, cBGround
    Text 44, 134, Str$(Level), CT, 3, 1, cEntry, cBGround
    
    Box Log_Box_x, Log_Box_y, 23, 23, 1, cEntry, cBGround
    If LogSweep Then
      Line Log_Box_x + 3, Log_Box_y + 3, Log_Box_x + 19, Log_Box_y + 19, 1, cEntry
      Line Log_Box_x + 3, Log_Box_y + 19, Log_Box_x + 19, Log_Box_y + 3, 1, cEntry
    EndIf
    Text MM.HRes, 180, "Log", RB, 1, 2, cEntry, cBGround
    
    ' setup the touch snsitive areas of the screen
    x1(5) = 0 : y1(5) = 0 : x2(5) = MM.HRes : y2(5) = 52
    x1(6) = 0 : y1(6) = 53 : x2(6) = MM.HRes : y2(6) = 112
    x1(7) = 101 : y1(7) = 107 : x2(7) = 230 : y2(7) = MM.VRes - 52
    x1(8) = 0 : y1(8) = 107 : x2(8) = 100 : y2(8) = MM.VRes - 52
    x1(9) = MM.HRes - 53 : y1(9) = Log_Box_y - 10 : x2(9) = MM.HRes : y2(9) = 190
    
    Do While Touch(x) <> -1 : WatchDog 5000 : Loop
    
    ' this will run the sweep and return if the screen has been touched
    SetLevel
    DoSweep SweepStart, SweepEnd, SweepTime
    
    i = CheckTouch(5, 12)
    If i > 0 Then SetTick 60000, SaveState, 2
    Select Case i
      Case 5
        KeyPadInput "Start(Hz)", "Greater Than End", "Zero Not Valid", SweepEnd, 1, SweepStart
        DrawModeButtons 3
      Case 6
        KeyPadInput " End (Hz)", "", "Less Than Start", 9999999, SweepStart, SweepEnd
        DrawModeButtons 3
      Case 7
        SweepTime = (SweepTime + 1) Mod 6
      Case 8
        KeyPadInput "Level (%)", "Max Is 100", "", 100, 0, Level
        DrawModeButtons 3
      Case 9
        LogSweep = Not LogSweep
      Case 10 To 12
        Mode = i - 10
        Exit Sub
    End Select
  Loop
End
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' utility routines
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' draw the mode buttons on the bottom of the main screen
Sub DrawModeButtons Sel
  Local i, fc, bc
  For i = 0 To 3
    If i = Sel Then
      fc = RGB(white) : bc = cSelected
    Else
      fc = RGB(0, 255, 204) : bc = cBGround
    EndIf
    Text i*80, 190, Str$(i), , 8, 1, fc, bc
    x1(10 + i) = i * 80 : y1(10 + i) = 190
    x2(10 + i) = (i + 1) * 80 : y2(10 + i) = 240
  Next
End Sub
  
  
  ' check if a touch sensitive area of the screen has been touched
Function CheckTouch(StartI, EndI)
  Local i, j, x, y
  x = Touch(x)
  If x = -1 Then CheckTouch = -1 : Exit Function
  y = Touch(y)
  For j = StartI To EndI
    If x >= x1(j) And x < x2(j) And y >= y1(j) And y < y2(j) Then
      CheckTouch = j
      Exit For
    EndIf
  Next j
End Function
  
  
  ' display the main frequency including the up/down arrows
Sub DisplayFreq Up, Dwn, ADigit
  Local String s
  Local i, p, bc
  Const GrpSpace = 12   ' space between groups of 3 digits
  Const HPos = 10       ' horiz position (for centering)
  
  Text HPos + 20,0, "Frequency (Hz)", ,1, 2 , cTitle, cBGround
  s = Str$(Freq, 7, 0, "0")
  p = HPos
  For i = 1 To 7
    If i = 8 - ADigit Then bc = cSelected Else bc = cBGround
    Text p, 30, Mid$(s, i, 1), , 3, 1, cEntry, bc
    x1(i) = p : y1(i) = 30 : x2(i) = p + 32 : y2(i) = 80
    p = p + 32
    If i = 1 Or i = 4 Then p = p + GrpSpace
  Next i
  Text MM.HRes - 10 + HPos, 3, Str$(Up * 2), R, 7, 1, cSpecial, cBGround
  Text MM.HRes - 10 + HPos, 55, Str$(1 + Dwn * 2), R, 7, 1, cSpecial, cBGround
  x1(8) = 320 - 77 + 13 + HPos : y1(8) = 3 : x2(8) = 320 : y2(8) = 53
  x1(9) = 320 - 77 + 13 + HPos : y1(9) = 55 : x2(9) = 320 : y2(9) = 105
End Sub
  
  
  ' display the main level including the up/down arrows
Sub DisplayLevel HPos, Up, Dwn, Hide
  Local String v = Str$(Level)
  Text HPos, 90, "Level", C,1, 2 , cTitle, cBGround
  If Hide Then
    v = "100"
    Box HPos - 48 - 50, 120, 58, 50, 0, 0, cBGround
    Box HPos + 48, 120, 58, 50, 0, 0, cBGround
    x1(14) = 0 : y1(14) = 0 : x2(14) = 0 : y2(14) = 0
    x1(15) = 0 : y1(15) = 0 : x2(15) = 0 : y2(15) = 0
  Else
    Text HPos - 48 - 50, 120, Str$(1 + Dwn * 2), , 7, 1, cSpecial, cBGround
    x1(14) = HPos - 48 - 50 : y1(14) = 120 : x2(14) = HPos- 48 : y2(14) = 170
    Text HPos + 48, 120, Str$(Up * 2), , 7, 1, cSpecial, cBGround
    x1(15) = HPos + 48 : y1(15) = 120 : x2(15) = HPos + 98 : y2(15) = 170
  EndIf
  If DisplayLevel.Last > Len(v) Then Text HPos, 120, "   ", C, 3, 1, cEntry, cBGround
  DisplayLevel.Last = Len(v)
  Text HPos, 120, v, C, 3, 1,  cEntry, cBGround
  x1(16) = HPos - 48 : y1(16) = 120 : x2(16) = HPos + 48 : y2(16) = 170
End Sub
  
  
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' interrupt subroutines
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' called on a touch interrupt
Sub MyInt
  TouchInt = 1
End Sub
  
  
  ' called on a timer 1 interrupt
Sub CursorOn
  Line MM.HRes - 25, 36, MM.HRes - 9, 36, 3, cEntry
  SetTick 500, CursorOff, 1
End Sub
  
  
  ' also called on a timer 1 interrupt
Sub CursorOff
  Line MM.HRes - 25, 36, MM.HRes - 9, 36, 3, cBGround
  SetTick 200, CursorOn, 1
End Sub
  
  
  ' called on a timer 2 interrupt
Sub SaveState
  VAR Save Freq, Level, Mode, ActiveDigit, SineAM, SweepStart, SweepEnd, SweepTime, LogSweep, CursorUpdateInterval, FreqSpan
  SetTick 0, 0, 2
End Sub
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' subroutines for handling the DDS hardware module
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  '
  ' AD9833 Control Bits
  '
  'CONST HLB = &H1000
  'CONST FSELECT = &H800
  'CONST PSELECT = &H400
  'CONST SLEEP1 = &H80
  '
  
  
  ' sets the frequency (Hz) and waveform for the main screen
  ' derived from an example written by Peter Mather (matherp on the Back Shed forum)
Sub SetFreq
  Local f, c, f1, f2
  Const oscillator = 25000000   'Crystal frequency
  Const p = &B1100000000000000
  Const B28 = &H2000
  Const RESET = &H100
  Const DEV2 = 8
  Const ADMODE = 2
  Const SLEEP12 = &H40
  Const OPBITEN = &H20
  
  f = ((Freq + FreqOffset) * (1 << 28)) \ oscillator
  f1 = (f And &H3FFF) Or &H4000
  f2 = (f >> 14) Or &H4000
  If Mode = 1 Then c = ADMODE
  If Mode = 2 Then c = SLEEP12 Or OPBITEN Or DEV2
  
  SPI Open 1000000, 2, 16
  Pin(FSY) = 0
  SPI write 5, c Or RESET Or B28, f1, f2, p, c
  Pin(FSY) = 1
  SPI Close
End Sub
  
  
  ' runs the sine wave sweep
Sub DoSweep st, en, swp
  Local f, f1, f2, Freq
  Local d(5 * 200)
  Local swp.steps(5) = (10000000, 315000, 107000, 36000, 16900, 13)
  Const oscillator = 25000000   'Crystal frequency
  Const p = &B1100000000000000
  Const RESET = &H100
  Const B28 = &H2000
  
  If st = 0 Or en = 0 Or en < st Then Exit Sub
  
  For i = 0 To 199*5 Step 5
    If LogSweep Then
      Freq = st + ((en - st) * ((Exp(((i / 5) / 199) * 2.30258) - 1) / 9))
    Else
      Freq = st + (((en - st) / 199) * (i / 5))
    EndIf
    f = (Freq * (1 << 28)) \ oscillator
    f1 = (f And &H3FFF) Or &H4000
    f2 = (f >> 14) Or &H4000
    d(i) = B28 : d(i+1) = f1 : d(i+2) = f2 : d(i+3) = p : d(i+4) = 0
  Next i
  
  CPU 48
  TouchInt = 0
  SetPin Peek(byte Peek(word &H9D000090) + 23), INTL, MyInt
  SPI Open swp.steps(swp), 2, 16
  Do
    WatchDog 6000
    Pulse TrigPin, 0.25
    Pin(FSY) = 0
    For i = 0 To 199*5 Step 5
      SPI write 5, d(i)
      If TouchInt Then Exit For
    Next i
    SPI Write 5, RESET Or B28, &H4000, &H4000, p, 0
    Pin(FSY) = 1
    Pause 5
  Loop Until TouchInt
  SPI Close
  SetPin 15, OFF
End Sub
  
  
  ' runs the sine wave AM modulation
  ' it will exit on any touch
Sub DoAM
  Local i = 0
  
  TouchInt = 0
  SetPin Peek(byte Peek(word &H9D000090) + 23), INTL, MyInt
  SPI Open 1000000, 2, 16
  Do
    WatchDog 1000
    Pin(CS) = 0
    SPI write 1, (Level/100) * (i * 211) Or &H1100
    Pin(CS) = 1
    i = Not i
  Loop Until TouchInt
  SPI Close
  SetPin 15, OFF
  SetLevel
End Sub
  
  
  
  ' sets the digital potentiometer (0-255)
  ' derived from an example written by Peter Mather (matherp on the Back Shed forum)
  ' note that when running on 3.3V the maximum undistorted level is 211
Sub SetLevel
  Local Integer x = (Level/100) * 211
  SPI Open 1000000, 2, 16
  Pin(CS) = 0
  SPI write 1, x Or &H1100
  Pin(CS) = 1
  SPI Close
End Sub
  
  
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' subroutines for handling the virtual keypad
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  
  
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' this displays a keypad for inputting a number
Sub KeyPadInput str As string, errhi As string, errlo As string, hi, lo, value
  Local i, b, nbr = 0
  Local String SCap(9) = ("7","8","9","4","5","6","1","2","3","0")
  Const bh = MM.VRes\5, bw = MM.HRes\4
  
  CLS cBGround
  Text 0, 10, str, , 1, 2, cTitle, cBGround
  Text MM.HRes - 5, 9, Str$(nbr), R, 2, 1, cEntry, cBGround
  CursorOn
  
  For i = 0 To 8
    DrawButton i, 0, bw/2 + bw * (i Mod 3), bh + bh * (i \ 3), bw - 4, bh - 4, cButton, SCap(i)
  Next i
  
  DrawButton i, 0, bw * 1.5, bh + bh * (i \ 3), bw - 4, bh - 4, cButton, "0"
  DrawButton 10, 0, 6, bh*4, 110, bh - 4, cDel, "DEL"
  DrawButton 11, 0, MM.HRes - 119, bh*4, 110, bh - 4, cSave, "SAVE"
  Do While Touch(x) <> -1 : WatchDog 5000 : Loop
  Do
    b = CheckButtonPress(0, 11)
    WatchDog 5000
    Select Case b
      Case 0 To 9
        If Len(Str$(nbr)) < 7 Then
          nbr = nbr * 10 + Val(SCap(b))
        EndIf
        Text MM.HRes - 5, 9, Str$(nbr), R, 2, 1, cEntry, cBGround
        Pause 150
        CheckButtonRelease b
      Case 10
        nbr = nbr \ 10
        Text MM.HRes - 5, 9, " " + Str$(nbr), R, 2, 1, cEntry, cBGround
        Pause 150
        CheckButtonRelease b
      Case 11
        CheckButtonRelease b
        SetTick 0, 0, 1
        If nbr > hi Then
          MessageBox "Error", errhi
        Else If nbr < lo Then
          MessageBox "Error", errlo
        Else
          value = nbr
        EndIf
        Exit Do
    End Select
  Loop
  CLS cBGround
End Sub
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' Draw buttons and get button presses
  '
  ' The subrouting DrawButton will draw a button (normally used when drawing
  ' the screen for input).
  '
  ' The function CheckButtonPress() will check if a button has been touched.
  ' If it has it will set it to selected (reverse video) and return with the
  ' button's number.
  '
  ' The subroutine CheckButtonRelease will wait for the touch to be released
  ' and will then draw the button as normal.
  '
  ' These routines use the global arrays key_coord() and key_caption() to
  ' track the coordinates and size of each button and save its caption.
  '
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' draw a button
Sub DrawButton n As Integer, mode As Integer, x As integer, y As integer, w As integer, h As integer, c As integer, s As string
  Local integer bc, fc
  
  If mode = 0 Then
    key_coord(n,0) = x : key_coord(n,1) = y : key_coord(n,2) = w : key_coord(n,3) = h
    key_coord(n,4) = c : key_caption(n) = s
  EndIf
  
  If mode > 1 Then
    bc = key_coord(n,4) : fc = 0    ' draw in reverse video if it is being touched
  Else
    bc = cBGround : fc = key_coord(n,4)    ' a normal (untouched) button
  EndIf
  
  RBox key_coord(n,0), key_coord(n,1), key_coord(n,2), key_coord(n,3), , key_coord(n,4), bc)
  Text key_coord(n,0) + key_coord(n,2)/2, key_coord(n,1) + key_coord(n,3)/2 + 4, key_caption(n), CM, 2, 1, fc, bc
End Sub
  
  
  ' check if a button has been touch and animate the button's image
  ' returns the button's number
Function CheckButtonPress(startn As Integer, endn As Integer) As Integer
  Local Integer xt, yellowt, n
  
  CheckButtonPress = -1
  If Touch(x) <> -1 Then
    ' we have a touch
    xt = Touch(x)
    yellowt = Touch(y)
    ' scan the array key_coord() to see if the touch was within the
    ' boundaries of a button
    For n = startn To endn
      If xt > key_coord(n,0) And xt < key_coord(n,0) + key_coord(n,2) And yellowt > key_coord(n,1) And yellowt < key_coord(n,1) + key_coord(n,3) Then
        ' we have a button press
        ' draw the button as pressed
        DrawButton n, 2
        CheckButtonPress = n
        Exit For
      EndIf
    Next n
  EndIf
End Function
  
  
  ' wait for the touch to be released and then draw the button as normal
Sub CheckButtonRelease n As integer
  ' if a button is currently down check if it has been released
  Do While Touch(x) <> -1 : WatchDog 5000 : Loop   ' wait for the button to be released
  DrawButton n, 1                  ' draw the button as normal (ie, not pressed)
End Sub
  
  
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' this handy routine draws a message box with an OK button
  ' then waits for the button to be touched
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Sub MessageBox s1 As String, s2 As String
  Local integer w
  If Len(s1) > Len(s2) Then w = Len(s1) Else w = Len(s2)
  w = w * 8     ' get the width of the text (used for the box width)
  
  ' draw the box and the message in it
  RBox MM.HRes/2 - w - 20, 60, w * 2 + 40, 130, , RGB(yellow), cBGround
  Text MM.HRes/2, 70, s1, CT, 1, 2, RGB(white), cBGround
  Text MM.HRes/2, 100, s2, CT, 1, 2, RGB(white), cBGround
  
  ' draw the OK button
  RBox 110, 140, 100, 34, , cButton, cBGround
  Text MM.HRes/2, 157, "OK", CM, 1, 2, cButton, cBGround
  
  ' wait for the button to be touched
  Do While Not (Touch(x) > 110 And Touch(x) < 210 And Touch(y) > 140 And Touch(y) < 180) : WatchDog 5000 : Loop
  
  ' draw the OK button as depressed
  RBox 110, 140, 100, 34, , cButton, cButton
  Text MM.HRes/2, 157, "OK", CM, 1, 2, 0, cButton
  
  ' wait for the touch to be removed
  Do While Touch(x) <> -1 : WatchDog 5000 : Loop
End Sub
  
SUB DrawIFAlignLine index, colour%
  LOCAL x1, x2, y1, y2
  LOCAL INTEGER i, avoidL, avoidR
  x1 = index * 319 / 79
  x2 = (index+1) * 319 / 79

  IF index MOD 16 = 0 THEN
    FOR i = 0 TO CursorNum-1
      IF CursorX(i) < 50 THEN avoidL = 1
      IF CursorX(i) > 320-25 THEN avoidR = 1
    NEXT i
    IF avoidR = 0 THEN Text 320, 0, STR$(CursorUpdateInterval) + "," + STR$(InvertVoltageReading), RT, 1, 1, RGB(WHITE), cBGround
    IF avoidL = 0 THEN
      Text 0, 0, STR$(FreqIterMin * 3.3, 1, 2), LT, 1, 1, RGB(WHITE), cBGround
      Text 0, 11, STR$((FreqIterMin + 1.0 / FreqIterScale) * 3.3, 1, 2), LT, 1, 1, RGB(WHITE), cBGround
    ENDIF
  ENDIF
  IF IFAlignData(index) < FreqIterMin THEN y1 = AlignGraphTop + AlignGraphHeight ELSE y1 = AlignGraphTop + AlignGraphHeight - LOG((IFAlignData(index)-FreqIterMin)*FreqIterScale*99+1)*(AlignGraphHeight/2)/2.303
  IF IFAlignData(index+1) < FreqIterMin THEN y2 = AlignGraphTop + AlignGraphHeight ELSE y2 = AlignGraphTop + AlignGraphHeight - LOG((IFAlignData(index+1)-FreqIterMin)*FreqIterScale*99+1)*(AlignGraphHeight/2)/2.303
  IF y1 < AlignGraphTop THEN y1 = AlignGraphTop
  IF y2 < AlignGraphTop THEN y2 = AlignGraphTop
  IF x1 > 0 THEN BOX x1+1, AlignGraphTop, x2 - x1, AlignGraphHeight+1, 1, RGB(BLACK), RGB(BLACK) ELSE BOX x1, AlignGraphTop, x2 - x1 + 1, AlignGraphHeight+1, 1, RGB(BLACK), RGB(BLACK)
  
  IF x1 <= 160 and x2 >= 160 THEN line 160, AlignGraphTop, 160, AlignGraphTop+AlignGraphHeight, 1, RGB(80,80,80)
  IF CursorNum > 0 and x1 = CursorX(0) THEN LINE CursorX(0), AlignGraphTop, CursorX(0), y1-1, 1, RGB(128,128,128)
  IF CursorNum > 1 and x1 = CursorX(1) THEN LINE CursorX(1), AlignGraphTop, CursorX(1), y1-1, 1, RGB(128,128,128)
  IF CursorNum > 2 and x1 = CursorX(2) THEN LINE CursorX(2), AlignGraphTop, CursorX(2), y1-1, 1, RGB(128,128,128)
  LINE x1, y1, x2, y2, 1, colour%
END SUB
  
SUB DrawCursor index, label$, avoid
  '  PRINT "DrawCursor" index label$
  LOCAL INTEGER x = index * 319 / 79
  LOCAL INTEGER avoidx
  IF avoid <> -1 THEN avoidx = avoid * 319 / 79 ELSE avoidx = -1000
  IF IFAlignData(index) < FreqIterMin THEN EXIT SUB
  LOCAL INTEGER y = AlignGraphTop + AlignGraphHeight - LOG((IFAlignData(index)-FreqIterMin)*FreqIterScale*99+1)*(AlignGraphHeight/2)/2.303
  LOCAL FLOAT frequency = Freq + FreqSpan * (index - 40) / 80
  LOCAL mytext$
  
  IF Freq >= 1000000 THEN
    mytext$ = STR$(frequency / 1000000.0, 1, 3) + "MHz"
  ELSE
    mytext$ = STR$(frequency / 1000.0, 1, 1) + "kHz"
  ENDIF

'  IF avoidx <> -1 THEN print x avoidx
  IF x < 34 THEN
    TEXT 0, 12, mytext$, LT, 1, 1, cEntry, cBGround    
  ELSEIF x > 319-34 THEN
    TEXT 319, 12, mytext$, RT, 1, 1, cEntry, cBGround    
  ELSE
    IF x < avoidx and avoidx - x < 60 THEN
      TEXT x, 12, mytext$, RT, 1, 1, cEntry, cBGround    
    ELSEIF x > avoidx AND x - avoidx < 60 THEN
      TEXT x, 12, mytext$, LT, 1, 1, cEntry, cBGround    
    ELSE
      TEXT x, 12, mytext$, CT, 1, 1, cEntry, cBGround    
    ENDIF
  ENDIF

  LINE x, AlignGraphTop, x, y-1, 1, RGB(128,128,128)
  IF label$ <> "" THEN
    IF x < 34 THEN
      TEXT 0, 0, label$, LT, 1, 1, RGB(255,128,128), cBGround
    ELSEIF x > 319-34 THEN
      TEXT 319, 0, label$, RT, 1, 1, RGB(255,128,128), cBGround
    ELSE
      IF x < avoidx and avoidx - x < 60 THEN
        TEXT x, 0, label$, RT, 1, 1, RGB(255,128,128), cBGround
      ELSEIF x > avoidx AND x - avoidx < 60 THEN
        TEXT x, 0, label$, LT, 1, 1, RGB(255,128,128), cBGround
      ELSE
        TEXT x, 0, label$, CT, 1, 1, RGB(255,128,128), cBGround
      ENDIF
    ENDIF
  ENDIF
  CursorX(CursorNum) = x
  CursorNum = CursorNum + 1
END SUB
  
SUB DrawCursors min!, max!
  LOCAL INTEGER i, j

  IF max! < 0.05 THEN EXIT SUB

  IF CursorNum > 0 THEN
    LOCAL INTEGER y
    FOR i = 0 TO CursorNum-1
      j = CursorX(i) * 79 / 319
      IF IFAlignData(j) >= FreqIterMin THEN
        y = AlignGraphTop + AlignGraphHeight - LOG((IFAlignData(j)-FreqIterMin)*FreqIterScale*99+1)*(AlignGraphHeight/2)/2.303
        LINE CursorX(i), AlignGraphTop, CursorX(i), y-1, 1, RGB(BLACK)
      ENDIF
    Next i
  ENDIF

  CursorNum = 0
  '  print "DrawCursors" min! max!
  BOX 0, 0, 320, AlignGraphTop, 1, RGB(BLACK), RGB(BLACK)
  FOR i = 0 to 80
    if IFAlignData(i) = max! THEN
      DrawCursor i, STR$(max! * 3.3, 1, 3) + "V", -1
      IF i > 0 THEN
        FOR j = i-1 TO 0 STEP -1
          IF IFAlignData(j) <= max! - (max! - min!) * (1-.707) THEN
            DrawCursor j, "-3dB", i
            EXIT FOR
          ENDIF
        NEXT j
      ENDIF
      IF i < 79 THEN
        FOR j = i+1 TO 79
          IF IFAlignData(j) <= max! - (max! - min!) * (1-.707) THEN
            DrawCursor j, "-3dB", i
            EXIT FOR
          ENDIF
        NEXT j
      ENDIF
      EXIT FOR
    ENDIF
  NEXT i
END SUB
  
SUB UpdateIFAlign
  IF FreqCount = 0 THEN
    FreqOffset = FreqSpan * (FreqIter - 40) / 80
    SetFreq
    FreqStarting = 1
  ELSE
    IF InvertVoltageReading = 1 THEN
      IFAlignData(FreqIter) = IFAlignData(FreqIter) + (3.3 - PIN(TrigPin)) * 3
    ELSEIF InvertVoltageReading = 1 THEN
      IFAlignData(FreqIter) = IFAlignData(FreqIter) + PIN(TrigPin) * 3
    ELSE
      IFAlignData(FreqIter) = IFAlignData(FreqIter) + PIN(TrigPin)
    ENDIF
  ENDIF
  FreqCount = FreqCount + 1
  IF FreqCount > FreqNumAvgs THEN
    FreqCount = 1
    IFAlignData(FreqIter) = (IFAlignData(FreqIter) * 3 / (3.3 * FreqNumAvgs) + OldIFAlignData) / 4
    IF FreqIter > 0 AND FreqIter < 80 THEN
      IF FreqStarting = 0 THEN
        IF FreqIter = 1 THEN DrawIFAlignLine 78, RGB(WHITE) ELSE DrawIFAlignLine FreqIter-2, RGB(WHITE)
      ELSE
        FreqStarting = 0
      ENDIF
      IF FreqIter = 1 THEN
        FreqIterScale = NewFreqIterScale
        FreqIterMin = NewFreqIterMin
      ENDIF
      DrawIFAlignLine FreqIter-1, RGB(BLUE)
    ENDIF
    FreqIter = FreqIter + 1
    IF FreqIter > 80 THEN
      FreqIter = 0
      NewFreqIterMin = 4
      LOCAL INTEGER i
      LOCAL FLOAT max = 0
      for i = 0 TO 80
        IF IFAlignData(i) < NewFreqIterMin THEN NewFreqIterMin = IFAlignData(i)
        IF IFAlignData(i) > max THEN max = IFAlignData(i)
      NEXT i
      NewFreqIterMin = NewFreqIterMin + (max - NewFreqIterMin) / 16
      IF max-NewFreqIterMin < 0.02 THEN NewFreqIterScale = 40.0 ELSE NewFreqIterScale = 0.8 / (max-NewFreqIterMin)
'      PRINT NewFreqIterMin max NewFreqIterScale
      CursorUpdateCounter = CursorUpdateCounter + 1
      IF CursorUpdateCounter >= CursorUpdateInterval THEN
        CursorUpdateCounter = 0
        DrawCursors NewFreqIterMin, max
      ENDIF
    ENDIF
    OldIFAlignData = IFAlignData(FreqIter)
    IFAlignData(FreqIter) = 0
    FreqOffset = FreqSpan * (FreqIter - 40) / 80
    SetFreq
  ENDIF
END SUB
  
  SUB ClearTouchZones start, num
    LOCAL INTEGER i
    FOR i = start to start + num
      x1(i) = 0 : x2(i) = 0 : y1(i) = 0 : y2(i) = 0
    NEXT i
  END SUB
  
  ' Inconsola
  ' Font type    : Truncated, no chard after Y and Y replaced with m
  ' Font size    : 24x32 pixels
  ' Memory usage : 5572 bytes
DefineFont #2
  3A202018 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00001800 3C00003C 003C0000 00003C00 3C00003C 001C0000 00001C00
  18000018 00180000 00001800 18000018 00180000 00001800 00000018 00000000
  00000000 3C000018 003C0000 00003C00 00000000 00000000 00000000 00000000
  00000000 0180C000 E101E0E1 E0F101E0 00E0E000 E000E060 C0C100C0 03808101
  00000003 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 E10080E1 80C10080 0080C300 C30080C3 F0FF0F00 01F0FF1F
  C30100C3 00830100 01008701 86010087 E0FF1F00 01E0FF1F 86030086 00060300
  03000603 0E03000E 000C0300 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 18000018 00FE0000 0380FF01 1807E09B 801807C0 07001807
  98030018 00F80300 0000FC00 1F00007F C01900C0 00E01800 18006018 60180460
  0FE01806 FF03C019 00FF0180 00001800 18000018 00000000 00000000 00000000
  00000000 01000000 E00700C0 60700770 0CE0300E 310CC030 80330E80 07007306
  C60300E7 000E0000 00000C00 1800001C 00300000 00E07100 E600F063 18C60038
  0118C601 86031886 F0030338 00E00106 00000000 00000000 00000000 00000000
  00000000 01002000 FE0300FC 008E0300 07000603 06030007 00860300 01008E03
  F80000DC 00F00000 0700F003 1C0740B8 701C0E60 0CE00E0E 031CC007 80030CC0
  0FC0070E FC07E01E 60F803F0 00000000 00000000 00000000 00000000 00000000
  00000000 00001800 3C00003C 003C0000 00001C00 1800000C 00180000 00003000
  00000060 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000100 0F000007 001C0080 00003800 70000070 00E00000 0100C000
  C00100C0 00800100 03008001 80030080 00800300 03008003 80010080 00800100
  0100C001 E00000C0 00E00000 00007000 38000078 001C0000 00000F00 03008007
  00000000 00008000 F00000E0 00780000 00001C00 0700000E 00070000 00800300
  01008003 C00100C0 00C00000 0000C000 E00000C0 00E00000 0000C000 C00100C0
  00C00100 03008001 80030080 00000700 1E00000E 003C0000 00007800 C00000F0
  00800000 00000000 00000000 00000000 00000000 1C000000 001C0000 04001C00
  1807101C F0C90770 00C0FF01 3C00003C 00760000 00006300 C10180E3 C08103C0
  00808000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 1C00001C 001C0000 00001C00
  1C00001C 001C0000 07F0FF07 1C00F0FF 001C0000 00001C00 1C00001C 001C0000
  00001C00 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00060000
  00000F00 0F00000F 00070000 00000300 06000006 000C0000 00001800 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 03E0FF03 0000E0FF 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000C00 1E00001E 001E0000 00000000 00000000 00000000 00000000 00000000
  00000000 00400000 0000E000 C00000E0 00C00100 03008001 00030080 00000700
  0E000007 000E0000 00001C00 3800001C 00380000 00003000 60000070 00E00000
  0100C000 800100C0 00800300 07008003 00010000 00000000 00000000 00000000
  00000000 00001000 FE00007C 00C70100 03808303 01078001 C00107C0 06C00006
  000EC000 E0000EE0 0EE0000E 000EE000 C00006E0 06C00006 0107C000 800103C0
  01808303 FE0000C7 007C0000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 3E00000E 00FE0100 0000C600 06000006 00060000 00000600
  06000006 00060000 00000600 06000006 00060000 00000600 06000006 00060000
  00000600 06000006 00060000 00000000 00000000 00000000 00000000 00000000
  00000000 00001000 FF0100FE 80870300 02C00107 0100C001 C00000C0 00C00100
  0300C001 80030080 00000700 1C00000E 00380000 00007000 C00100E0 00800300
  07008003 FF072000 C0FF07C0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 FF0300FC 00070700 00800302 01008001 80010080 00800300
  1E000007 007C0000 00007F00 03008007 C0010080 00C00100 0100C001 C00102C0
  07800302 FF038087 00FC0100 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 0F000007 000F0000 00001F00 3700003F 00670000 00006700
  C70100C7 00870100 07000703 07060007 E0FF0F00 0FE0FF0F 0700E0FF 00070000
  00000700 07000007 00070000 00000000 00000000 00000000 00000000 00000000
  00000000 03000000 FF03C0FF 000003C0 03000003 00030000 00000300 07000007
  FF0700FE 80870700 00C00107 0000C001 E00000C0 00E00000 0000E000 C00102C0
  07C0010E FF038087 00FE0100 00000000 00000000 00000000 00000000 00000000
  00000000 00000800 FF00007F 80C101C0 03808003 00070000 00000700 06000006
  FF06003E 80C70700 07808107 0006C001 C00006C0 06C00006 0007C000 C00103C0
  01C08103 FF0080C3 007E0000 00000000 00000000 00000000 00000000 00000000
  00000000 07000000 FF07C0FF C00100C0 00800300 03008003 00070000 00000700
  0E00000E 000C0000 00001C00 3800001C 00380000 00003800 70000070 00700000
  0000E000 E00000E0 00C00100 00000000 00000000 00000000 00000000 00000000
  00000000 00001800 FF0100FE 80830300 07800103 0107C001 C00103C0 01808303
  FE0000C7 007C0000 0300FF01 030780C7 C0010780 0EC00006 000EE000 C00006C0
  07C00107 FF038083 00FE0080 00000000 00000000 00000000 00000000 00000000
  00000000 00001000 FF0100FE 80870300 07800307 0106C001 C00006C0 06C00006
  0007C000 C00107C0 03C08303 FC00C0FE C02000C0 00C00100 0100C001 800300C0
  03800302 FE07000F 00F80100 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00180000 00003C00 3C00003C 00000000 00000000
  00000000 00000000 00000000 00000000 00180000 00003C00 3C00003C 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 000C0000 00001E00 1E00001E 00000000 00000000
  00000000 00000000 00000000 00000000 000C0000 00001E00 1E00001E 000E0000
  00000600 0C00000C 00180000 00003000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00200000 0300E000 800F00E0 00003E00
  E00300F8 00800F00 1E00001E 800F0000 00E00300 0000F000 1F00007C C0070000
  00E00100 0000E000 00000020 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 F0FF0700 00F0FF07
  00000000 00000000 00000000 FF070000 F0FF07F0 00F0FF07 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 0E000008 800F0000 00E00300 0000F800
  0F00003E C0030000 00E00000 0300E000 800F00C0 00003E00 F0010078 00C00700
  1C00001F 00180000 00000000 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 FF00003E C0FF0180 03C08107 0001E000 E00000E0 00E00000
  0100E000 C00300E0 00800700 0E000007 000C0000 00000C00 1C00001C 00000000
  00000000 0C000000 001E0000 00001E00 0000001E 00000000 00000000 00000000
  00000000 00000000 FF00001C 80E70100 07C08103 0006E000 60000E60 0CE0030C
  3C1CE01F 60701C60 18606018 601C6060 E0701C60 0CE0791C 0E0CE03F 00000E00
  03000007 C0030080 C0FF00C0 00807F00 00000000 00000000 00000000 00000000
  00000000 00002000 30000030 00300000 00007800 78000078 00FC0000 0000CC00
  860100CE 00860100 03008701 03030003 80FF0380 0680FF07 010EC001 C0000EC0
  1CE0000C 001CE000 70001870 00000000 00000000 00000000 00000000 00000000
  00000000 0F000000 FF0F00FE 80070C00 0CC0010C 000CC001 C0010CC0 0CC0010C
  FF0F8003 00FE0F00 0C80FF0F 010CC003 E0000CC0 0CE0000C 000CE000 E0000CE0
  0CC0010C FF0FC007 00FC0F80 00000000 00000000 00000000 00000000 00000000
  00000000 00000800 FF01007F E0C103C0 07E08007 000E7000 00000E00 0C00000E
  000C0000 00000C00 0C00000C 000C0000 00000C00 0E00000E 00070000 40000700
  03E08003 FF00C0C1 007F0080 00000000 00000000 00000000 00000000 00000000
  00000000 0F000000 FF0F00FC 800F0C00 0C80030C 010CC001 E0000CC0 0CE0000C
  000CE000 E0000CE0 0C60000C 000C6000 E0000CE0 0CE0000C 010CC000 C0010CC0
  0C80030C FE0F000F 00F80F00 00000000 00000000 00000000 00000000 00000000
  00000000 0F000000 FF0FE0FF 00000EE0 0E00000E 000E0000 00000E00 0E00000E
  000E0000 00FF0F00 0E00FF0F 000E0000 00000E00 0E00000E 000E0000 00000E00
  0E00000E FF0F0000 C0FF0FC0 00000000 00000000 00000000 00000000 00000000
  00000000 07000000 FF07C0FF 000007C0 07000007 00070000 00000700 07000007
  FF070000 00FF0700 0700FF07 00070000 00000700 07000007 00070000 00000700
  07000007 00070000 00000700 00000000 00000000 00000000 00000000 00000000
  00000000 00000400 FF00807F E0E001C0 03708003 00072080 00000700 06000006
  00060000 00000E00 0E00000E 030EF003 300006F0 07300006 00073000 30800330
  0130C003 FF00F0E0 803F00E0 00000000 00000000 00000000 00000000 00000000
  00000000 0E000000 000EE000 E0000EE0 0EE0000E 000EE000 E0000EE0 0EE0000E
  000EE000 E0FF0FE0 0EE0FF0F 000EE000 E0000EE0 0EE0000E 000EE000 E0000EE0
  0EE0000E 000EE000 E0000EE0 00000000 00000000 00000000 00000000 00000000
  00000000 07000000 FF0780FF 00300080 00003000 30000030 00300000 00003000
  30000030 00300000 00003000 30000030 00300000 00003000 30000030 00300000
  00003000 FF070030 80FF0780 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 FF00E0FF 000700E0 00000700 07000007 00070000 00000700
  07000007 00070000 00000700 07000007 00070000 00000700 06000006 000E0000
  0F000E04 F807001C 00F80300 00000000 00000000 00000000 00000000 00000000
  00000000 0CA00002 030CC001 80070C80 0C00070C 1C0C000E 00380C00 0C00700C
  C00D00E0 00E00F00 0E00E00F 380C0070 003C0C00 0C001C0C 0F0C000E 00070C00
  0C80030C 010CC001 F0000CE0 00000000 00000000 00000000 00000000 00000000
  00000000 07000000 00070000 00000700 07000007 00070000 00000700 07000007
  00070000 00000700 07000007 00070000 00000700 07000007 00070000 00000700
  07000007 FF070000 E0FF07E0 00000000 00000000 00000000 00000000 00000000
  00000000 0C000000 000E6000 E0000EE0 0FE0010F 830FE001 60830FE0 0C60C70D
  EE0C60C6 606C0C60 0C60780C 300C6038 60100C60 0C60000C 000C6000 60000C60
  0C60000C 000C6000 60000C60 00000000 00000000 00000000 00000000 00000000
  00000000 0E000000 000EE000 E0000FE0 0FE0800F C00DE080 E0C00DE0 0CE0E00C
  700CE060 E0380CE0 0CE0380C 1C0CE01C E00E0CE0 0CE00E0C 030CE007 E0030CE0
  0CE0010C 000CE001 E0000CE0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 FF0100FE 80830380 0EC00107 000EC000 E0000CE0 1C60001C
  001C6000 70001C70 1C70001C 001C7000 60001C70 0E60000C 000EE000 C00107E0
  03C08107 FF0180C7 00FE0000 00000000 00000000 00000000 00000000 00000000
  00000000 0F000000 FF0F00FF C0030E80 0EC0010E 000EE000 E0000EE0 0EE0000E
  010EE000 80FF0FC0 0F00FF0F 000E00FC 00000E00 0E00000E 000E0000 00000E00
  0E00000E 000E0000 00000E00 00000000 00000000 00000000 00000000 00000000
  00000000 00000800 FF00007F C0C101C0 07E08003 00076000 70000670 0E30000E
  000E3000 38000E38 0E38000E 000E3800 30000E38 07300006 00073000 60000370
  01E08003 FF00C0E3 007E0080 00001C00 0E00000C F00F0000 00F00700 00000000
  00000000 0F000000 FF0F00FF C0030E80 0EC0010E 000EE000 E0000EE0 0EE0000E
  030EC001 80FF0FC0 0F00FF0F 0C0E00FC 000E0E00 0E00070E 030E0007 80030E80
  0EC0010E 000EC001 E0000EE0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 FF00807F E0C001C0 03608003 80034080 00800300 01008003
  F80000E0 007E0000 00801F00 0100C007 F00000E0 00700000 00007000 70000270
  07E00007 FF01E0C1 00FF00C0 00000000 00000000 00000000 00000000 00000000
  00000000 0F000000 FF0FF0FF 001800F0 00001800 18000018 00180000 00001800
  18000018 00180000 00001800 18000018 00180000 00001800 18000018 00180000
  00001800 18000018 00180000 00000000 00000000 00000000 00000000 00000000
  00000000 0E000000 000C6000 60000C60 0C60000C 000C6000 60000C60 0C60000C
  000C6000 60000C60 0C60000C 000C6000 60000C60 0E60000C 000EE000 E0000EE0
  07C00107 FF03C083 00FE0080 00000000 00000000 00000000 00000000 00000000
  00000000 1C000000 000C7000 E0000E60 06E0000E 0107C000 C00107C0 03800103
  83038081 00830180 0100C301 C60000C7 00E60000 0000EE00 7C00006C 007C0000
  00003800 38000038 00100000 00000000 00000000 00000000 00000000 00000000
  00000000 18000000 10183000 30101C30 1C30181C 380C7018 603C0C60 0C603C0C
  6C0E606C 606E0E60 06C06606 C606C0C6 C0C706C0 07C08307 8307C083 80830380
  03800103 01038001 80010380 00000000 00000000 00000000 00000000 00000000
  00000000 0E200000 0107C000 800107C0 03808303 C7010083 00CE0100 0000EE00
  7C00007C 00380000 00003800 7C00007C 00EE0000 0100CE00 830300C7 80830380
  07C00107 000EC001 F0000EE0 00000000 00000000 00000000 00000000 00000000
  00000000 00000000 00000000 00000000 00000000 00000000 40200000 0FF0F90E
  1E0F70FB 381C0E30 0E381C0E 1C0E381C 381C0E38 0E381C0E 1C0E381C 381C0E38
  0E381C0E 1C0E381C 381C0E38 00381C0E 00000000 00000000 00000000 00000000
  00000000
End DefineFont
  
  
  ' Font type    : Numeric (10 characters)
  ' Font size    : 32x50 pixels
  ' Memory usage : 2004 bytes
DefineFont #3
  0A303220 00000000 00000000 00F00F00 00F81F00 00FE7F00 00FFFF00 00FFFF01
  801FF803 C00FF003 C007E007 E003C007 E003C00F F001800F F001800F F001801F
  F800001F F800001F F800001F F800001F FC00003F 7C00003E 7C00003E 7C00003E
  7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003F
  F800001F F800001F F800001F F800001F F001800F F001800F F001800F E003C007
  E003C007 C007E003 C00FF003 801FF801 00FFFF00 00FEFF00 00FC7F00 00F81F00
  00E00700 00000000 00000000 00000000 00000000 00000000 00F00000 00F00100
  00F00300 00F00700 00F00F00 00F01F00 00F03F00 00F07F00 00F0FF01 00F0FD03
  00F0F10F 00F0E10F 00F0810F 00F0010F 00F0010C 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100 00F00100
  00F00100 00F00100 00F00100 00000000 00000000 00000000 00000000 00F00F00
  00FE7F00 00FFFF01 C0FFFF03 E0FFFF03 F00FE007 F003C00F F001800F F800001F
  F800001F F800003E 7C00003E 7C00003E 7C000000 7C000000 7C000000 78000000
  F8000000 F8010000 F0030000 F0070000 E00F0000 C01F0000 803F0000 007F0000
  00FE0000 00FC0100 00F80300 00F00700 00E00F00 00C01F00 00803F00 00007F00
  0000FE00 0000FC01 0000F803 0000F007 0000E007 0000C00F 0000800F 0000801F
  FCFFFF3F FCFFFF3F FCFFFF7F FCFFFF7F FCFFFF7F 00000000 00000000 00000000
  00000000 00F00F00 00FC3F00 00FEFF00 00FFFF01 80FFFF03 C01FE003 C00FC007
  E0078007 E003800F E003000F F003000F F001000F F0030000 E0030000 E0030000
  E0070000 C00F0000 C01F0000 803F0000 00FF0700 00FF0700 00FF0700 80FF0700
  C01F0000 E00F0000 E0030000 F0010000 F0000000 F8000000 F8000000 FC000000
  FC000000 FC000000 FC00001F FC00001F F800001F F801001F F801800F F003800F
  F007C00F E00FE007 C03FFC03 80FFFF01 00FFFF00 00FC7F00 00F00F00 00000000
  00000000 00000000 00000000 000F0000 001F0000 001F0000 003F0000 003F0000
  007F0000 007F0000 00FF0000 00FF0100 00FF0100 00FF0300 00DF0700 00DF0700
  009F0F00 009F0F00 001F1F00 001F3E00 001F7E00 001F7C00 001FF800 001FF800
  001FF001 001FE003 001FE003 001FC007 001FC007 001F800F 001F801F 001F001F
  001F003E FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F 001F0000 001F0000
  001F0000 001F0000 001F0000 001F0000 001F0000 001F0000 001F0000 001F0000
  001F0000 00000000 00000000 00000000 00000000 F0FFFF01 F0FFFF01 F0FFFF01
  F0FFFF01 F0FFFF01 0000E003 0000C003 0000C003 0000C003 0000C007 0000C007
  0000C007 0000C007 00008007 0000800F 00F0800F 00FE8F0F 80FFBF0F C0FFFF0F
  E0FFFF0F F007FF1F F003E01F F801C01F F800001F F8000000 FC000000 7C000000
  7C000000 7C000000 7C000000 7C000000 7C000000 7C000000 F800003F F800003F
  F800003F F000001F F001001F E003801F E007C00F C01FE007 80FFFF03 00FFFF03
  00FEFF01 00F87F00 00E00F00 00000000 00000000 00000000 00000000 00F00F00
  00FE3F00 00FFFF00 80FFFF01 C0FFFF03 E007E007 F003C007 F001C00F F800800F
  F800801F F800801F 0000001F 0000001F 0000003F 0000003F 0000003F 00F0033F
  00FC1F3F 00FE7F3F 80FFFF3F C0FFFF3F E00FF83F F003E03F F003E03F F801C03F
  F800803F F800803F 7C00003F 7C00003F 7C00003F 7C00003F 7C00003F 7C00003F
  7C00001F 7800801F F800801F F800801F F800800F F001C00F F003F007 E00FF803
  C0FFFF01 80FFFF00 00FF7F00 00FC1F00 00F00300 00000000 00000000 00000000
  00000000 FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FCFFFF3F FC000000 FC000000
  F8010000 F0010000 E0030000 E0030000 C0070000 800F0000 001F0000 003E0000
  007C0000 00FC0000 00F80000 00F00000 00F00100 00F00100 00E00300 00E00300
  00E00300 00C00700 00C00700 00800F00 00800F00 00800F00 00001F00 00001F00
  00001F00 00001F00 00003E00 00003E00 00003E00 00003E00 00003E00 00007C00
  00007C00 00007C00 00007C00 00007C00 0000F800 0000F800 0000F800 00000000
  00000000 00000000 00000000 00F00F00 00FE7F00 80FFFF00 80FFFF01 C01FF803
  E007E007 E003C007 F003C00F F001800F F001800F F001800F F001800F F001800F
  F001800F E003C007 E003C007 C007E003 8007F001 003FFC00 00FF7F00 00FC3F00
  00FE7F00 00FFFF00 C0FFFF03 E00FF007 F003C00F F001800F F801801F F800001F
  F800001F 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E 7C00003E F800001F
  F800001F F001800F F003800F E007C007 C00FF003 80FFFF01 00FFFF00 00FC3F00
  00F00F00 00000000 00000000 00000000 00000000 00F00F00 00FC3F00 00FFFF00
  80FFFF01 C0FFFF03 E01FF803 E007E007 F007E007 F003C00F F803C00F F801801F
  F801801F FC00001F FC00001F FC00001F FC00001F FC00001F FC00001F FC01801F
  FC01801F FC03C00F FC03C00F FC07E007 FC07E007 FC1FF803 FCFFFF01 FCFFFF00
  FCFEFF00 FCF87F00 FCE01F00 FC000000 FC000000 FC000000 FC000000 F8010000
  F801003F F001003F F003001F E007801F E00FC00F C03FE00F 80FFFF07 00FFFF03
  00FEFF01 00F8FF00 00E03F00 00000000 00000000
End DefineFont
  
  
  ' Pointers as a font
  ' Font type    : Triangles (4 characters)
  ' Font size    : 52x50 pixels
  ' Memory usage : 1304 bytes
DefineFont #7
  04303234 70000000 00000000 00000700 00000000 000000D8 800D0000 00000000
  00008C01 18000000 000000C0 00060300 00000000 00006030 03060000 00000000
  00306000 0C000000 00008001 18C00000 00000000 00C00018 80010000 0000000C
  60003000 03000000 00000600 00600000 00000030 00030006 C0000000 00001800
  01000C00 01000080 000C0080 00180000 0000C000 06000003 30000000 00600000
  00000600 00000003 30000060 000C0000 00800100 0000C000 18000018 C0000000
  00800100 00000C00 00000030 00030060 00060000 00006000 06003000 03000000
  00C00000 00180000 0000000C 80018001 0C000000 00001800 03C00000 00000000
  00300006 60000000 00000006 60000300 00000000 00000C30 80010000 000000C0
  00181800 00000000 000080C1 300C0000 00000000 00006300 06000000 FFFFFF7F
  FFF7FFFF FFFFFFFF FFFF7FFF F7FFFFFF FFFFFFFF 0030FFFF 00000000 00000063
  18060000 00000000 0080C100 0C000000 0000000C C0800100 00000000 00000618
  00030000 00000060 00033000 06000000 00003000 01600000 00000080 0018000C
  C0000000 0000C000 0C001800 01000000 00600080 00300000 00000006 30000003
  60000000 00000300 00000600 00000018 800100C0 000C0000 00000C00 00008001
  180000C0 00060000 00000300 00006000 03000030 00060000 00300000 00006000
  0C008001 18000000 00C00000 00C00000 00000018 8001000C 60000000 00003000
  03000600 00000000 00600030 00030000 00000006 C0001800 01000000 00000C80
  010C0000 00000080 0018C000 06000000 00000003 30600000 00000000 00000603
  30000000 00000060 008C0100 00000000 0000C018 D8000000 00000000 00800D00
  00000000 00000070 00070000 00000000 00007000 07000000 00000000 00F80000
  00000000 0000800F FC010000 00000000 00C01F00 03000000 000000FE E03F0000
  00000000 0000FF07 7F000000 000000F0 80FF0F00 00000000 0000F8FF FF1F0000
  000000C0 00FCFF01 3F000000 0000E0FF FEFF0300 00000000 00F0FF7F FF070000
  000000FF F8FFFF00 0F000000 0080FFFF FFFF0100 000000FC C0FFFF1F FF030000
  0000FEFF FFFF3F00 070000E0 00FFFFFF FF7F0000 0000F0FF FFFFFF0F FF000080
  00F8FFFF FFFF1F00 0100C0FF FCFFFFFF FF3F0000 00E0FFFF FFFFFF03 7F0000FE
  F0FFFFFF FFFF0700 0000FFFF FFFFFFFF FF0F00F8 80FFFFFF FFFFFF01 1F00FCFF
  FFFFFFFF FFFF03C0 00FEFFFF FFFFFF3F FF07E0FF FFFFFFFF FFFF7F00 0FF0FFFF
  FFFFFFFF FFFF80FF F8FFFFFF FFFFFF1F FFC1FFFF FFFFFFFF FFFF3FFC E3FFFFFF
  FFFFFFFF FF7FFEFF FFFFFFFF FFFFFFF7 7FFFFFFF FFFFFFFF FFFFF7FF FFFFFFFF
  FFFFFF3F FFE3FFFF FFFFFFFF FFFF1FFE C1FFFFFF FFFFFFFF FF0FFCFF FFFFFFFF
  FFFFFF80 07F8FFFF FFFFFFFF FF7F00FF F0FFFFFF FFFFFF03 3F00FEFF FFFFFFFF
  FFFF01E0 00FCFFFF FFFFFF1F FF00C0FF F8FFFFFF FFFF0F00 0080FFFF FFFFFF7F
  FF0700F0 00FFFFFF FFFF3F00 0300E0FF FEFFFFFF FF1F0000 00C0FFFF FFFFFF01
  0F0000FC 80FFFFFF FFFF0000 0000F8FF FFFFFF07 7F000000 00F0FFFF FFFF0300
  000000FE E0FFFF3F FF010000 0000FCFF FFFF1F00 000000C0 00F8FFFF FF0F0000
  000080FF F0FF7F00 07000000 0000FFFF FF3F0000 000000E0 00FEFF03 1F000000
  0000C0FF FCFF0100 00000000 0080FF0F FF000000 000000F8 00FF0700 00000000
  0000F07F FE030000 00000000 00E03F00 01000000 000000FC C01F0000 00000000
  0000F800 0F000000 00000080 00700000 00000000 00000007
End DefineFont
  
  
  ' Special symbols as a font
  ' Font type    : Symbols (4 characters)
  ' Font size    : 80x50 pixels
  ' Memory usage : 2004 bytes
DefineFont #8
  04303250 00000000 00000000 FF7F0000 FFFFFFFF FEFFFFFF 00000040 00000000
  00400200 00000000 02000000 00000040 00000000 00400200 00000000 02000000
  00000040 00000000 00400200 00000000 02000000 00000040 00000000 00400200
  00000000 0200C007 F0010040 E00E0000 00400200 0000B807 0200601C 1C060040
  70180000 00400200 00000C0E 02003038 0E0C0040 18300000 00400200 0000061C
  02001870 07180040 1C600000 00400200 00000318 02000CE0 03300040 0CC00000
  00400200 00000330 020006C0 01300040 06C00180 00400200 01800160 02000680
  01600040 078001C0 00400200 01C00060 02000380 00E00040 030003C0 00400200
  03E000C0 02800300 00C00040 01000360 01400280 076000C0 02800100 00800140
  01000660 01400280 06700080 02C00000 00800340 00000E30 034002C0 0C300000
  02E00000 00000340 00000C18 03400260 1C180000 02600000 00000640 0000181C
  06400230 180C0000 02300000 00000C40 0000300E 9C400238 30060000 821C0000
  0000F840 0000E007 F840820F C0030000 020F0000 00000040 00000000 00400200
  00000000 02000000 00000040 00000000 00400200 00000000 02000000 00000040
  00000000 00400200 00000000 02000000 00000040 00000000 00400200 00000000
  02000000 FFFFFF7F FFFFFFFF 0000FEFF 00000000 00000000 00000000 00000000
  FF7F0000 FFFFFFFF FEFFFFFF 00000040 00000000 00400200 00000000 02000000
  00000040 00000000 00400200 00000000 02000000 00000040 00000000 00400200
  00000000 02000000 00000040 00000000 00400200 00000000 02000000 00000040
  00000000 00400200 00004000 02000004 E0000040 000E0000 00400200 0000B001
  0200001B B0010040 001B0000 00400200 00001803 02008031 18030040 80310000
  00400200 00000C06 0200C060 0C060040 C0600000 00400200 0000060C 020060C0
  060C0040 60C00000 00400200 01000318 02003080 03180040 30800100 00400200
  03800130 02001800 01300040 18000380 00400200 06C00060 02000C00 00600040
  0C0006C0 00400200 0C6000C0 02000600 00C00040 06000C60 01400200 18300080
  02000300 00800140 03001830 83418201 30180000 02830100 00008341 01003018
  C6400283 600C0000 02C60000 0000C640 0000600C 6C4002C6 C0060000 026C0000
  00006C40 0000C006 3840026C 80030000 02380000 00001040 00000001 00400210
  00000000 02000000 00000040 00000000 00400200 00000000 02000000 00000040
  00000000 00400200 00000000 02000000 00000040 00000000 00400200 00000000
  02000000 00000040 00000000 00400200 00000000 02000000 FFFFFF7F FFFFFFFF
  0000FEFF 00000000 00000000 00000000 00000000 FF7F0000 FFFFFFFF FEFFFFFF
  00000040 00000000 00400200 00000000 02000000 00000040 00000000 00400200
  00000000 02000000 00000040 00000000 00400200 00000000 02000000 00000040
  00000000 00400200 00000000 02000000 00000040 00000000 07400200 0100FFFF
  02C0FFFF FFFF0740 FFFF0100 064002C0 01000300 02C00080 03000640 00800100
  064002C0 01000300 02C00080 03000640 00800100 064002C0 01000300 02C00080
  03000640 00800100 064002C0 01000300 02C00080 03000640 00800100 064002C0
  01000300 02C00080 03000640 00800100 064002C0 01000300 02C00080 03000640
  00800100 064002C0 01000300 02C00080 03000640 00800100 064002C0 01000300
  02C00080 03000640 00800100 064002C0 01000300 02C00080 03000640 00800100
  064002C0 01000300 02C00080 03000640 00800100 064002C0 01000300 02C00080
  03000640 00800100 064002C0 01000300 02C00080 03000640 00800100 FE4002C0
  FFFF0300 02FF0080 0300FE40 0080FFFF 004002FF 00000000 02000000 00000040
  00000000 00400200 00000000 02000000 00000040 00000000 00400200 00000000
  02000000 00000040 00000000 00400200 00000000 02000000 00000040 00000000
  00400200 00000000 02000000 FFFFFF7F FFFFFFFF 0000FEFF 00000000 00000000
  00000000 00000000 FF7F0000 FFFFFFFF FEFFFFFF 00000040 00000000 00400200
  00000000 02000000 00000040 00000000 00400200 00000000 02000000 00000040
  00000000 00400200 00000000 02000000 00000040 00000000 00400200 00700000
  02000000 00000040 000000D8 00400200 00880000 02600030 01000041 0038000C
  00410250 00040300 02900048 02008041 00480004 80400290 00040600 0298004C
  0400C040 01C40002 40400288 00020400 02080184 08004040 01840002 60400208
  00020800 020C0184 08002040 01820003 20400204 01011000 02040382 10003040
  02020101 10400204 01011000 02040202 30001040 02020101 10400204 01012000
  02060202 20000840 02038101 08400202 83006000 02020601 40000840 04018200
  0C400202 82004000 02020401 C0000440 04018200 04400202 82008000 02038401
  80000240 84004200 02400201 44008001 02018800 00010340 88004400 01400201
  44000003 8201C800 00820140 58002C00 00408200 28000086 82005000 00C40040
  70002800 00408200 10000068 02002000 00380040 00000000 00400200 00000000
  02000000 00000040 00000000 00400200 00000000 02000000 00000040 00000000
  00400200 00000000 02000000 00000040 00000000 00400200 00000000 02000000
  FFFFFF7F FFFFFFFF 0000FEFF 00000000 00000000
End DefineFont
